;NSIS script for Osdag (Open Steel Design and Graphics) installer
;-----------------------------------------------------------------------
# Notes:
	;Currently, the code to install miniconda silently is included as comments in this file.
	;However, to install the python dependencies, the system environment variables must be updated.
	;This can be done by refreshing the sys-env-var inside the setup before calling the batch file to install the dependencies. Jayant couldn't find a reliable code to do this and the idea is therefore kept on hold.
;-----------------------------------------------------------------------

; Include scripts

	;NSIS Modern User Interface
	!include "MUI2.nsh"
	
	;Set the environment variables
	!include "x64.nsh"	

;-----------------------------------------------------------------------
;General Installer Settings

	;Properly display all languages (Installer will not work on Windows 95, 98 or ME!)
	Unicode true  
	
	;Declare name of installer file
	Name "Osdag"
	OutFile "Osdag_setup_x64.exe"

	;Default installation Drive
	InstallDir "$DESKTOP\Osdag"
  
	;Get installation folder from registry if available
	InstallDirRegKey HKLM "Software\Osdag" ""
	
	;Add Osdag branding and remove NSIS, in the installer
	BrandingText "Osdag Beta v2017.08.a.e2dd"	

;-----------------------------------------------------------------------
;Declare Variables (User defined)

	Var Start_menu_folder
	;Var TEMP_Osdag # Folder used to temporarily store files
	Var Osdag_workspace # Default Osdag_workspace location	

;-----------------------------------------------------------------------
;Interface Settings

	RequestExecutionLevel admin ;Require admin rights on NT6+ (When UAC is turned on)
	
	!define MUI_ABORTWARNING
	
	;Show all languages, despite user's codepage
	!define MUI_LANGDLL_ALLLANGUAGES

;-----------------------------------------------------------------------
;Modern User Interface parameters

	;Define Windows Registry Entries
	;Language Selection Dialog Settings - Remember the installer language
	!define MUI_LANGDLL_REGISTRY_ROOT "HKLM"   
	!define MUI_LANGDLL_REGISTRY_KEY "Software\Osdag" 
	!define MUI_LANGDLL_REGISTRY_VALUENAME "Installer Language"
  
	;Start Menu Registry
	!define MUI_STARTMENUPAGE_REGISTRY_ROOT "HKLM" 
	!define MUI_STARTMENUPAGE_REGISTRY_KEY "Software\Osdag" 
	!define MUI_STARTMENUPAGE_REGISTRY_VALUENAME "Start Menu Folder"
 
	;Set logo to Setup file  
	!define MUI_ICON osdag_logo.ico  
	
	# Uncomment the below 5 lines, when Miniconda setup is embedded	
	;!define MUI_COMPONENTSPAGE_TEXT_TOP " "
	;!define MUI_COMPONENTSPAGE_TEXT_COMPLIST "Please uncheck Miniconda only if it is already installed."
	;!define MUI_COMPONENTSPAGE_TEXT_INSTTYPE " "
	;!define MUI_COMPONENTSPAGE_TEXT_DESCRIPTION_TITLE "Component Description"
	;!define MUI_COMPONENTSPAGE_TEXT_DESCRIPTION_INFO "Please see the README.txt file if Miniconda is already installed."
	;The component descriptions are declared, in MUI_FUNCTION_DESCRIPTION_BEGIN, after all the sections.

;-----------------------------------------------------------------------
;Pages

	;Installer Design (/Layout)
	!define MUI_WELCOMEPAGE_TEXT "This Setup will guide you through the installation of Osdag.$\r$\n$\r$\nIt is recommended that you close all other applications before starting the Setup. This will make it possible to update relevant system files without restarting your computer. $\r$\n $\r$\nThe Setup also installs some dependencies that are required to run Osdag. $\r$\n $\r$\n Click Next to continue." 
	!insertmacro MUI_PAGE_WELCOME  
	!insertmacro MUI_PAGE_LICENSE "License.txt"	
	;!insertmacro MUI_PAGE_COMPONENTS # Uncomment when Miniconda setup is embedded
	!insertmacro MUI_PAGE_DIRECTORY
	
	;Following lines of code are for selecting default Osdag_workspace location
	!define $Osdag_workspace "$INSTDIR\Osdag_workspace"
	!define MUI_PAGE_HEADER_SUBTEXT "Select the default location for Osdag workspace"
	!define MUI_DIRECTORYPAGE_TEXT_TOP "Osdag workspace is the directory where all the design files will be saved. To select a different location, click Browse and specify a different lcoation. Click Next to continue."
	!define MUI_DIRECTORYPAGE_VARIABLE $Osdag_workspace
	!insertmacro MUI_PAGE_DIRECTORY
	;!define $Osdag_workspace MUI_DIRECTORYPAGE_VARIABLE
	
	!insertmacro MUI_PAGE_STARTMENU Application $Start_menu_folder
	!insertmacro MUI_PAGE_INSTFILES
	!insertmacro MUI_PAGE_FINISH
	
	;Uninstaller Design (/Layout)
	!define MUI_WELCOMEPAGE_TEXT "This Setup will guide you through the uninstallation of Osdag. $\r$\n$\r$\nBefore starting the uninstallation, please make sure that Osdag is not running. $\r$\n$\r$\nAfter uninstalling Osdag, you need to separately uninstall Miniconda2 from your computer. $\r$\n$\r$\nClick Next to continue."
	!insertmacro MUI_UNPAGE_WELCOME
	!insertmacro MUI_UNPAGE_CONFIRM
	!insertmacro MUI_UNPAGE_INSTFILES
	!insertmacro MUI_UNPAGE_FINISH
  
	;-----------------------------------------------------------------------
	;Languages

	!insertmacro MUI_LANGUAGE "English"
	;!insertmacro MUI_LANGUAGE "German"   


;-----------------------------------------------------------------------
; User defined functions

Function create_install_log
	
	;Create osdag.config and osdag-installation.log files
	;Set the location to copy files temporarily, if required
	;StrCpy $TEMP_Osdag "$TEMP\Osdag"

	;Create config file (with default path values)
	
	FileOpen $0 "$INSTDIR\Osdag.config" w
	FileWrite $0 "[default_workspace]"
	FileWrite $0 "$\r$\npath1 = $Osdag_workspace\"
	FileWrite $0 "$\r$\n$\r$\n[installation_dir]"
	FileWrite $0 "$\r$\npath1 = $INSTDIR\"
	FileWrite $0 "$\r$\n$\r$\n[wkhtml_path]"
	FileWrite $0 "$\r$\npath1 = $PROGRAMFILES32\wkhtmltopdf\bin\wkhtmltopdf.exe"	
	FileClose $0

	;Create installation-log file 
	FileOpen $1 "$INSTDIR\osdag-installation.log" w
	FileWrite $1 "[TEMP_folder]"
	FileWrite $1 "$\r$\n$TEMP_Osdag\"
	FileClose $1

FunctionEnd

Function create_osdag_launcher

	;Create Osdag.bat (This batch file, when invoked, will start Osdag.)
	FileOpen $0 "$INSTDIR\Osdag.bat" w
	FileWrite $0 "@setlocal enableextensions"
	FileWrite $0 "$\r$\n@cd /d '%~dp0'"
	FileWrite $0 "$\r$\n@prompt Osdag:"
	FileWrite $0 "$\r$\n@cd /d '$INSTDIR'"
	FileWrite $0 "$\r$\n@REM Please do not close this window."
	FileWrite $0 "$\r$\n@REM Please wait while Osdag is being initialized..."
	FileWrite $0 "$\r$\n@REM Osdag will display python console output in this window."
	FileWrite $0 "$\r$\n@echo off"
	FileWrite $0 "$\r$\nstart pythonw osdagMainPage.py"
	FileWrite $0 "$\r$\n@REM Osdag was closed."
	FileClose $0

	;Create shortcut for Osdag.bat
	CreateShortCut "$INSTDIR\Osdag.lnk" "$INSTDIR\Osdag.bat" "" "$INSTDIR\ResourceFiles\images\Osdag Icon.ico" "" SW_SHOWMINIMIZED

FunctionEnd

;-----------------------------------------------------------------------
;Sections (/pages /steps) in the Installer

Section "Osdag" SEC01

	;This section installs the Osdag python code
	SetOutPath "$INSTDIR"

	;Copy Osdag python code to the installation folder
	File /r Osdag\*.*

	;Save installation-folder to Registry Key
	WriteRegStr HKLM "Software\Osdag" "" $INSTDIR

	;Create uninstaller
		;Refine this uninstaller using
		;http://nsis.sourceforge.net/Uninstall_only_installed_files
	WriteUninstaller "$INSTDIR\Uninstall.exe"

	;Create osdag.config and osdag-installation.log files
	Call create_install_log
	Call create_osdag_launcher

	;Create shortcuts
	;Create Start Menu shortcut
	!insertmacro MUI_STARTMENU_WRITE_BEGIN Application	
	CreateDirectory "$SMPROGRAMS\$Start_menu_folder"	
	CreateShortCut "$SMPROGRAMS\$Start_menu_folder\Osdag.lnk" "$INSTDIR\Osdag.bat" "" "$INSTDIR\ResourceFiles\images\Osdag Icon.ico" "" SW_SHOWMINIMIZED
	CreateShortCut "$SMPROGRAMS\$Start_menu_folder\Uninstall.lnk" "$INSTDIR\Uninstall.exe"
	!insertmacro MUI_STARTMENU_WRITE_END
	
	;Create Desktop shortcut	
	CreateShortCut "$DESKTOP\Osdag.lnk" "$INSTDIR\Osdag.bat" "" "$INSTDIR\ResourceFiles\images\Osdag Icon.ico" "" SW_SHOWMINIMIZED
	
SectionEnd

Section "Dependencies" SEC02
	
	;This section silently installs wkhtmltopdf (and Miniconda) and then installs the python dependencies that are required to run Osdag
	
	;Copy Setup of Miniconda2 to temporary folder
	# Uncomment below 2 lines when Miniconda setup is embedded
	;SetOutPath "$TEMP"
	;File Miniconda2-latest-Windows-x86.exe 
	
	;Invoke Miniconda installer to run silently with the following parameters:
	;Install for: [All Users (requires admin privileges)]
	;Destination Folder: ["C:\ProgramData\Miniconda2"]
	;Add Anaconda to PATH environment variable (for all users, so it is the System PATH)
	;Register Anaconda as my default Python 2.7	(for all users)
	# Uncomment below line when Miniconda setup is embedded
	;ExecWait "$TEMP\Miniconda2-latest-Windows-x86.exe /InstallationType=AllUsers /AddToPath=1 /RegisterPython=1 /S /D=$PROGRAMFILES32\Miniconda2"	
	
	;Install wkhtmltopdf silently	
	SetOutPath "$TEMP"
	File wkhtmltox-0.12.4_msvc2015-win32.exe
	nsExec::ExecToLog "$TEMP\wkhtmltox-0.12.4_msvc2015-win32.exe /S /D=$PROGRAMFILES32\wkhtmltopdf"	
	
	;Copy dependencies to installation folder
	SetOutPath "$INSTDIR\dependencies"
	File /r dependencies\*.*
	
	;Run a batch file to install the (conda and pip) dependencies
	nsExec::ExecToLog "$INSTDIR\dependencies\install_osdag_dependencies.bat"

	SetOutPath "$PROGRAMFILES32\Miniconda2\Lib\site-packages\"
	File /r cairo\*.*
		
	;Do NOT Cleanup the Minicond2 setup and the python package tar files
	;These may be used by the required in the future.
	;Uncomment the below two lines if Cleanup is preferred.
	;Delete "$TEMP\Miniconda2-latest-Windows-x86.exe"
	;RMDir /r "@INSTDIR\dependencies\"
	
SectionEnd

# Uncomment below macro-set when Miniconda setup is embedded
;!insertmacro MUI_FUNCTION_DESCRIPTION_BEGIN
	;This macro sets the description for each component in the installation. This should be placed AFTER the section IDs have been defined, and hence after the Sections.
	;!insertmacro MUI_DESCRIPTION_TEXT ${SEC01} "This installs the Osdag engine."
	;!insertmacro MUI_DESCRIPTION_TEXT ${SEC02} "Miniconda is a python package manager essential to run Osdag. If you already have Miniconda installed, uncheck this and see the README.txt file for instructions."
;!insertmacro MUI_FUNCTION_DESCRIPTION_END

Section "Uninstall" 

	;Uninstaller Section
	
	;Remove Start Menu shortcuts
	!insertmacro MUI_STARTMENU_GETFOLDER Application $Start_menu_folder    
	Delete "$SMPROGRAMS\$Start_menu_folder\Uninstall.lnk"
	RMDir /r "$SMPROGRAMS\$Start_menu_folder\"
	
	;Remove Desktop shortcut (if it exists)
	Delete "$DESKTOP\Osdag.lnk"
  
	;Remove installed files
	Delete "$INSTDIR\Uninstall.exe" ;Explicitly delete the Uninstaller
	RMDir /r "$INSTDIR\*"
	
	;Remove Miniconda2 Setup stored in the TEMP folder
	# Uncomment below line when Miniconda setup is embedded
	;Delete "$TEMP\Miniconda2-latest-Windows-x86.exe"
  
	;Remove Registry Entry 
	;Add /isempty parameter if the Osdag registry entry needs to be kept for future installation
	;Ideally, a clean uninstall also removes the registry entries.
	DeleteRegKey HKLM "Software\Osdag"
  
SectionEnd

;-----------------------------------------------------------------------
;Update built-in functions with custom calls

	;Installer functions
	;Function .onInit
		;!insertmacro MUI_LANGDLL_DISPLAY    
	;FunctionEnd

	;Uninstaller functions
	;Function un.onInit
		;!insertmacro MUI_UNGETLANGUAGE  
	;FunctionEnd 
 
Function .onInstSuccess

	;Function that calls a messagebox when installation finished correctly
	MessageBox MB_OK "You have successfully installed Osdag. Use the desktop or Start Menu shortcut to start the program."
	
FunctionEnd
 
Function un.onUninstSuccess

	;Function that calls a messagebox when uninstallation finished correctly
	MessageBox MB_OK "You have successfully uninstalled Osdag."
	
FunctionEnd