# -*- coding: utf-8 -*-
from distutils.core import setup

packages = \
['pdflatex']

package_data = \
{'': ['*']}

install_requires = \
['attrs>=18.2,<19.0']

setup_kwargs = {
    'name': 'pdflatex',
    'version': '0.1.3',
    'description': 'Simple wrapper to calling pdflatex',
    'long_description': "# Overview\n\nThis is a simple module to execute pdflatex in an easy and clean way.\nThe pdflatex command line utility by default generates a lot of output and can create many files.\n\n## Instantiation\n\nThe PDFLaTeX class can be instantiated directly or through helpers:\n\n- **from_texfile(filename)**\n- **from_binarystring(binstr, jobname)**\n- **from_jinja2_template(jinja2_template, jobname = None, \\*\\*render_kwargs)**\n\njobname is any string that can be used to create a valid filename;\n\n\n## Examples:\n\nIn all the following examples, no files are left on the filesystem,\nunless requested with the *keep_pdf_file* and *keep_log_file* parameters\nto the *create_pdf* method.\n\n### Create PDF from .tex file\n\n    from pdflatex import PDFLaTeX\n\n    pdfl = PDFLaTeX.from_texfile('my_file.tex')\n    pdf, log, completed_process = pdfl.create_pdf(keep_pdf_file=True, keep_log_file=True)\n\nThe function **create_pdf** returns 3 results:\n1. The pdf file in a binary string;\n2. The log file generated by pdflatex as text;\n3. An instance of subprocess.CompleteProcess with the results of the pdflatex execution.\n\nAlso, **create_pdf** takes a few parameters:\n1. keep_pdf_file: an optional boolean. Default to False. If True a pdf file is maintained. Its location and name depends on the value of the -output-directory and -jobname parameters. It is also derived from the tex filename or the jinja2 template filename if no parameter is given;\n2. keep_log_file: same thing, for the log file.\n3. env: a default ENV mapping for the execution of pdflatex. You probably want to skip this.\n\n\n### Create PDF from Jinja2 Template\n\n    import os\n    import pdflatex\n    import jinja2 \n    \n    env = pdflatex.JINJA2_ENV\n    env['loader'] = jinja2.FileSystemLoader(os.path.abspath('.'))\n    env = jinja2.Environment(**env)\n    template = env.get_template('jinja.tex')\n    pdfl = pdflatex.PDFLaTeX.from_jinja2_template(template, data='Hello world!')\n    pdf, log, cp = pdfl.create_pdf()\n\nQuite self explanatory, just note that pdflatex includes a dictionary\nJINJA2_ENV with a suggestion of environment parameters for you to use with\njinja2 and LaTeX. It os defined as:\n\n    JINJA2_ENV = {'block_start_string': '\\BLOCK{',\n                  'block_end_string': '}',\n                  'variable_start_string': '\\VAR{',\n                  'variable_end_string': '}',\n                  'comment_start_string': '\\#{',\n                  'comment_end_string': '}',\n                  'line_statement_prefix': '%%',\n                  'line_comment_prefix': '%#',\n                  'trim_blocks': True,\n                  'autoescape': False }\n\n\n### Create PDF file from binary string:\n\n    import pdflatex\n\n    with open('my_file.tex', 'rb') as f:\n        pdfl = pdflatex.PDFLaTeX.from_binarystring(f.read(), 'my_file')\n    pdf, log, cp = pdfl.create_pdf()",
    'author': 'mbello',
    'author_email': 'mbello@users.noreply.github.com',
    'url': 'https://github.com/mbello/pdflatex',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
