
# This file helps to compute a version number in source trees obtained from
# git-archive tarball (such as those provided by githubs download-from-tag
# feature). Distribution tarballs (built by setup.py sdist) and build
# directories (produced by setup.py build) will contain a much shorter file
# that just contains the computed version number.

# This file is released into the public domain. Generated by
# versioneer-0.17 (https://github.com/warner/python-versioneer)

u"""Git implementation of _version.py."""

from __future__ import absolute_import
from __future__ import print_function
from __future__ import unicode_literals
from __future__ import division
from builtins import int
from builtins import open
from future import standard_library
standard_library.install_aliases()
import errno
import os
import re
import subprocess
import sys
from io import open


def get_keywords():
    u"""Get the keywords needed to look up the version information."""
    # these strings will be replaced by git during git-archive.
    # setup.py/versioneer.py will grep for the variable names, so they must
    # each be defined on a line of their own. _version.py will just call
    # get_keywords().
    git_refnames = u"$Format:%d$"
    git_full = u"$Format:%H$"
    git_date = u"$Format:%ci$"
    keywords = {u"refnames": git_refnames, u"full": git_full, u"date": git_date}
    return keywords


class VersioneerConfig(object):
    u"""Container for Versioneer configuration parameters."""


def get_config():
    u"""Create, populate and return the VersioneerConfig() object."""
    # these strings are filled in when 'setup.py versioneer' creates
    # _version.py
    cfg = VersioneerConfig()
    cfg.VCS = u"git"
    cfg.style = u"pep440"
    cfg.tag_prefix = u"v"
    cfg.parentdir_prefix = u"PyLaTeX-"
    cfg.versionfile_source = u"pylatex/_version.py"
    cfg.verbose = False
    return cfg


class NotThisMethod(Exception):
    u"""Exception raised if a method is not valid for the current scenario."""


LONG_VERSION_PY = {}
HANDLERS = {}


def register_vcs_handler(vcs, method):  # decorator
    u"""Mark a method as the handler for a particular VCS."""
    def decorate(f):
        u"""Store f in HANDLERS[vcs][method]."""
        if vcs not in HANDLERS:
            HANDLERS[vcs] = {}
        HANDLERS[vcs][method] = f
        return f
    return decorate


def run_command(commands, args, cwd=None, verbose=False, hide_stderr=False,
                env=None):
    u"""Call the given command(s)."""
    assert isinstance(commands, list)
    p = None
    for c in commands:
        try:
            dispcmd = unicode([c] + args)
            # remember shell=False, so use git.cmd on windows, not just git
            p = subprocess.Popen([c] + args, cwd=cwd, env=env,
                                 stdout=subprocess.PIPE,
                                 stderr=(subprocess.PIPE if hide_stderr
                                         else None))
            break
        except EnvironmentError:
            e = sys.exc_info()[1]
            if e.errno == errno.ENOENT:
                continue
            if verbose:
                print(u"unable to run %s" % dispcmd)
                print(e)
            return None, None
    else:
        if verbose:
            print(u"unable to find command, tried %s" % (commands,))
        return None, None
    stdout = p.communicate()[0].strip()
    if sys.version_info[0] >= 3:
        stdout = stdout.decode()
    if p.returncode != 0:
        if verbose:
            print(u"unable to run %s (error)" % dispcmd)
            print(u"stdout was %s" % stdout)
        return None, p.returncode
    return stdout, p.returncode


def versions_from_parentdir(parentdir_prefix, root, verbose):
    u"""Try to determine the version from the parent directory name.

    Source tarballs conventionally unpack into a directory that includes both
    the project name and a version string. We will also support searching up
    two directory levels for an appropriately named parent directory
    """
    rootdirs = []

    for i in xrange(3):
        dirname = os.path.basename(root)
        if dirname.startswith(parentdir_prefix):
            return {u"version": dirname[len(parentdir_prefix):],
                    u"full-revisionid": None,
                    u"dirty": False, u"error": None, u"date": None}
        else:
            rootdirs.append(root)
            root = os.path.dirname(root)  # up a level

    if verbose:
        print(u"Tried directories %s but none started with prefix %s" %
              (unicode(rootdirs), parentdir_prefix))
    raise NotThisMethod(u"rootdir doesn't start with parentdir_prefix")


@register_vcs_handler(u"git", u"get_keywords")
def git_get_keywords(versionfile_abs):
    u"""Extract version information from the given file."""
    # the code embedded in _version.py can just fetch the value of these
    # keywords. When used from setup.py, we don't want to import _version.py,
    # so we do it with a regexp instead. This function is not used from
    # _version.py.
    keywords = {}
    try:
        f = open(versionfile_abs, u"r")
        for line in f.readlines():
            if line.strip().startswith(u"git_refnames ="):
                mo = re.search(ur'=\s*"(.*)"', line)
                if mo:
                    keywords[u"refnames"] = mo.group(1)
            if line.strip().startswith(u"git_full ="):
                mo = re.search(ur'=\s*"(.*)"', line)
                if mo:
                    keywords[u"full"] = mo.group(1)
            if line.strip().startswith(u"git_date ="):
                mo = re.search(ur'=\s*"(.*)"', line)
                if mo:
                    keywords[u"date"] = mo.group(1)
        f.close()
    except EnvironmentError:
        pass
    return keywords


@register_vcs_handler(u"git", u"keywords")
def git_versions_from_keywords(keywords, tag_prefix, verbose):
    u"""Get version information from git keywords."""
    if not keywords:
        raise NotThisMethod(u"no keywords at all, weird")
    date = keywords.get(u"date")
    if date is not None:
        # git-2.2.0 added "%cI", which expands to an ISO-8601 -compliant
        # datestamp. However we prefer "%ci" (which expands to an "ISO-8601
        # -like" string, which we must then edit to make compliant), because
        # it's been around since git-1.5.3, and it's too difficult to
        # discover which version we're using, or to work around using an
        # older one.
        date = date.strip().replace(u" ", u"T", 1).replace(u" ", u"", 1)
    refnames = keywords[u"refnames"].strip()
    if refnames.startswith(u"$Format"):
        if verbose:
            print(u"keywords are unexpanded, not using")
        raise NotThisMethod(u"unexpanded keywords, not a git-archive tarball")
    refs = set([r.strip() for r in refnames.strip(u"()").split(u",")])
    # starting in git-1.8.3, tags are listed as "tag: foo-1.0" instead of
    # just "foo-1.0". If we see a "tag: " prefix, prefer those.
    TAG = u"tag: "
    tags = set([r[len(TAG):] for r in refs if r.startswith(TAG)])
    if not tags:
        # Either we're using git < 1.8.3, or there really are no tags. We use
        # a heuristic: assume all version tags have a digit. The old git %d
        # expansion behaves like git log --decorate=short and strips out the
        # refs/heads/ and refs/tags/ prefixes that would let us distinguish
        # between branches and tags. By ignoring refnames without digits, we
        # filter out many common branch names like "release" and
        # "stabilization", as well as "HEAD" and "master".
        tags = set([r for r in refs if re.search(ur'\d', r)])
        if verbose:
            print(u"discarding '%s', no digits" % u",".join(refs - tags))
    if verbose:
        print(u"likely tags: %s" % u",".join(sorted(tags)))
    for ref in sorted(tags):
        # sorting will prefer e.g. "2.0" over "2.0rc1"
        if ref.startswith(tag_prefix):
            r = ref[len(tag_prefix):]
            if verbose:
                print(u"picking %s" % r)
            return {u"version": r,
                    u"full-revisionid": keywords[u"full"].strip(),
                    u"dirty": False, u"error": None,
                    u"date": date}
    # no suitable tags, so version is "0+unknown", but full hex is still there
    if verbose:
        print(u"no suitable tags, using unknown + full revision id")
    return {u"version": u"0+unknown",
            u"full-revisionid": keywords[u"full"].strip(),
            u"dirty": False, u"error": u"no suitable tags", u"date": None}


@register_vcs_handler(u"git", u"pieces_from_vcs")
def git_pieces_from_vcs(tag_prefix, root, verbose, run_command=run_command):
    u"""Get version from 'git describe' in the root of the source tree.

    This only gets called if the git-archive 'subst' keywords were *not*
    expanded, and _version.py hasn't already been rewritten with a short
    version string, meaning we're inside a checked out source tree.
    """
    GITS = [u"git"]
    if sys.platform == u"win32":
        GITS = [u"git.cmd", u"git.exe"]

    out, rc = run_command(GITS, [u"rev-parse", u"--git-dir"], cwd=root,
                          hide_stderr=True)
    if rc != 0:
        if verbose:
            print(u"Directory %s not under git control" % root)
        raise NotThisMethod(u"'git rev-parse --git-dir' returned error")

    # if there is a tag matching tag_prefix, this yields TAG-NUM-gHEX[-dirty]
    # if there isn't one, this yields HEX[-dirty] (no NUM)
    describe_out, rc = run_command(GITS, [u"describe", u"--tags", u"--dirty",
                                          u"--always", u"--long",
                                          u"--match", u"%s*" % tag_prefix],
                                   cwd=root)
    # --long was added in git-1.5.5
    if describe_out is None:
        raise NotThisMethod(u"'git describe' failed")
    describe_out = describe_out.strip()
    full_out, rc = run_command(GITS, [u"rev-parse", u"HEAD"], cwd=root)
    if full_out is None:
        raise NotThisMethod(u"'git rev-parse' failed")
    full_out = full_out.strip()

    pieces = {}
    pieces[u"long"] = full_out
    pieces[u"short"] = full_out[:7]  # maybe improved later
    pieces[u"error"] = None

    # parse describe_out. It will be like TAG-NUM-gHEX[-dirty] or HEX[-dirty]
    # TAG might have hyphens.
    git_describe = describe_out

    # look for -dirty suffix
    dirty = git_describe.endswith(u"-dirty")
    pieces[u"dirty"] = dirty
    if dirty:
        git_describe = git_describe[:git_describe.rindex(u"-dirty")]

    # now we have TAG-NUM-gHEX or HEX

    if u"-" in git_describe:
        # TAG-NUM-gHEX
        mo = re.search(ur'^(.+)-(\d+)-g([0-9a-f]+)$', git_describe)
        if not mo:
            # unparseable. Maybe git-describe is misbehaving?
            pieces[u"error"] = (u"unable to parse git-describe output: '%s'"
                               % describe_out)
            return pieces

        # tag
        full_tag = mo.group(1)
        if not full_tag.startswith(tag_prefix):
            if verbose:
                fmt = u"tag '%s' doesn't start with prefix '%s'"
                print(fmt % (full_tag, tag_prefix))
            pieces[u"error"] = (u"tag '%s' doesn't start with prefix '%s'"
                               % (full_tag, tag_prefix))
            return pieces
        pieces[u"closest-tag"] = full_tag[len(tag_prefix):]

        # distance: number of commits since tag
        pieces[u"distance"] = int(mo.group(2))

        # commit: short hex revision ID
        pieces[u"short"] = mo.group(3)

    else:
        # HEX: no tags
        pieces[u"closest-tag"] = None
        count_out, rc = run_command(GITS, [u"rev-list", u"HEAD", u"--count"],
                                    cwd=root)
        pieces[u"distance"] = int(count_out)  # total number of commits

    # commit date: see ISO-8601 comment in git_versions_from_keywords()
    date = run_command(GITS, [u"show", u"-s", u"--format=%ci", u"HEAD"],
                       cwd=root)[0].strip()
    pieces[u"date"] = date.strip().replace(u" ", u"T", 1).replace(u" ", u"", 1)

    return pieces


def plus_or_dot(pieces):
    u"""Return a + if we don't already have one, else return a ."""
    if u"+" in pieces.get(u"closest-tag", u""):
        return u"."
    return u"+"


def render_pep440(pieces):
    u"""Build up version string, with post-release "local version identifier".

    Our goal: TAG[+DISTANCE.gHEX[.dirty]] . Note that if you
    get a tagged build and then dirty it, you'll get TAG+0.gHEX.dirty

    Exceptions:
    1: no tags. git_describe was just HEX. 0+untagged.DISTANCE.gHEX[.dirty]
    """
    if pieces[u"closest-tag"]:
        rendered = pieces[u"closest-tag"]
        if pieces[u"distance"] or pieces[u"dirty"]:
            rendered += plus_or_dot(pieces)
            rendered += u"%d.g%s" % (pieces[u"distance"], pieces[u"short"])
            if pieces[u"dirty"]:
                rendered += u".dirty"
    else:
        # exception #1
        rendered = u"0+untagged.%d.g%s" % (pieces[u"distance"],
                                          pieces[u"short"])
        if pieces[u"dirty"]:
            rendered += u".dirty"
    return rendered


def render_pep440_pre(pieces):
    u"""TAG[.post.devDISTANCE] -- No -dirty.

    Exceptions:
    1: no tags. 0.post.devDISTANCE
    """
    if pieces[u"closest-tag"]:
        rendered = pieces[u"closest-tag"]
        if pieces[u"distance"]:
            rendered += u".post.dev%d" % pieces[u"distance"]
    else:
        # exception #1
        rendered = u"0.post.dev%d" % pieces[u"distance"]
    return rendered


def render_pep440_post(pieces):
    u"""TAG[.postDISTANCE[.dev0]+gHEX] .

    The ".dev0" means dirty. Note that .dev0 sorts backwards
    (a dirty tree will appear "older" than the corresponding clean one),
    but you shouldn't be releasing software with -dirty anyways.

    Exceptions:
    1: no tags. 0.postDISTANCE[.dev0]
    """
    if pieces[u"closest-tag"]:
        rendered = pieces[u"closest-tag"]
        if pieces[u"distance"] or pieces[u"dirty"]:
            rendered += u".post%d" % pieces[u"distance"]
            if pieces[u"dirty"]:
                rendered += u".dev0"
            rendered += plus_or_dot(pieces)
            rendered += u"g%s" % pieces[u"short"]
    else:
        # exception #1
        rendered = u"0.post%d" % pieces[u"distance"]
        if pieces[u"dirty"]:
            rendered += u".dev0"
        rendered += u"+g%s" % pieces[u"short"]
    return rendered


def render_pep440_old(pieces):
    u"""TAG[.postDISTANCE[.dev0]] .

    The ".dev0" means dirty.

    Eexceptions:
    1: no tags. 0.postDISTANCE[.dev0]
    """
    if pieces[u"closest-tag"]:
        rendered = pieces[u"closest-tag"]
        if pieces[u"distance"] or pieces[u"dirty"]:
            rendered += u".post%d" % pieces[u"distance"]
            if pieces[u"dirty"]:
                rendered += u".dev0"
    else:
        # exception #1
        rendered = u"0.post%d" % pieces[u"distance"]
        if pieces[u"dirty"]:
            rendered += u".dev0"
    return rendered


def render_git_describe(pieces):
    u"""TAG[-DISTANCE-gHEX][-dirty].

    Like 'git describe --tags --dirty --always'.

    Exceptions:
    1: no tags. HEX[-dirty]  (note: no 'g' prefix)
    """
    if pieces[u"closest-tag"]:
        rendered = pieces[u"closest-tag"]
        if pieces[u"distance"]:
            rendered += u"-%d-g%s" % (pieces[u"distance"], pieces[u"short"])
    else:
        # exception #1
        rendered = pieces[u"short"]
    if pieces[u"dirty"]:
        rendered += u"-dirty"
    return rendered


def render_git_describe_long(pieces):
    u"""TAG-DISTANCE-gHEX[-dirty].

    Like 'git describe --tags --dirty --always -long'.
    The distance/hash is unconditional.

    Exceptions:
    1: no tags. HEX[-dirty]  (note: no 'g' prefix)
    """
    if pieces[u"closest-tag"]:
        rendered = pieces[u"closest-tag"]
        rendered += u"-%d-g%s" % (pieces[u"distance"], pieces[u"short"])
    else:
        # exception #1
        rendered = pieces[u"short"]
    if pieces[u"dirty"]:
        rendered += u"-dirty"
    return rendered


def render(pieces, style):
    u"""Render the given version pieces into the requested style."""
    if pieces[u"error"]:
        return {u"version": u"unknown",
                u"full-revisionid": pieces.get(u"long"),
                u"dirty": None,
                u"error": pieces[u"error"],
                u"date": None}

    if not style or style == u"default":
        style = u"pep440"  # the default

    if style == u"pep440":
        rendered = render_pep440(pieces)
    elif style == u"pep440-pre":
        rendered = render_pep440_pre(pieces)
    elif style == u"pep440-post":
        rendered = render_pep440_post(pieces)
    elif style == u"pep440-old":
        rendered = render_pep440_old(pieces)
    elif style == u"git-describe":
        rendered = render_git_describe(pieces)
    elif style == u"git-describe-long":
        rendered = render_git_describe_long(pieces)
    else:
        raise ValueError(u"unknown style '%s'" % style)

    return {u"version": rendered, u"full-revisionid": pieces[u"long"],
            u"dirty": pieces[u"dirty"], u"error": None,
            u"date": pieces.get(u"date")}


def get_versions():
    u"""Get version information or return default if unable to do so."""
    # I am in _version.py, which lives at ROOT/VERSIONFILE_SOURCE. If we have
    # __file__, we can work backwards from there to the root. Some
    # py2exe/bbfreeze/non-CPython implementations don't do __file__, in which
    # case we can only use expanded keywords.

    cfg = get_config()
    verbose = cfg.verbose

    try:
        return git_versions_from_keywords(get_keywords(), cfg.tag_prefix,
                                          verbose)
    except NotThisMethod:
        pass

    try:
        root = os.path.realpath(__file__)
        # versionfile_source is the relative path from the top of the source
        # tree (where the .git directory might live) to this file. Invert
        # this to find the root from __file__.
        for i in cfg.versionfile_source.split(u'/'):
            root = os.path.dirname(root)
    except NameError:
        return {u"version": u"0+unknown", u"full-revisionid": None,
                u"dirty": None,
                u"error": u"unable to find root of source tree",
                u"date": None}

    try:
        pieces = git_pieces_from_vcs(cfg.tag_prefix, root, verbose)
        return render(pieces, cfg.style)
    except NotThisMethod:
        pass

    try:
        if cfg.parentdir_prefix:
            return versions_from_parentdir(cfg.parentdir_prefix, root, verbose)
    except NotThisMethod:
        pass

    return {u"version": u"0+unknown", u"full-revisionid": None,
            u"dirty": None,
            u"error": u"unable to compute version", u"date": None}
